<?php

namespace Modules\Appointment\Repositories\Eloquents;

use App\User;
use App\Models\Language;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Modules\Appointment\Entities\Social;
use Modules\CourseSetting\Entities\Category;
use Modules\SystemSetting\Entities\Currency;
use App\Repositories\Eloquents\BaseRepository;
use Modules\CourseSetting\Entities\CourseLevel;
use Modules\Appointment\Entities\InstructorSocial;
use Modules\Appointment\Entities\InstructorEducation;
use Modules\Appointment\Entities\InstructorCertificate;
use Modules\Appointment\Entities\InstructorWorkExperience;
use Modules\Appointment\Entities\InstructorTeachingCategory;
use Modules\Appointment\Entities\InstructorTeachingLanguage;
use Modules\Appointment\Repositories\Interfaces\AppointmentRepositoryInterface;

class AppointmentRepository extends BaseRepository implements AppointmentRepositoryInterface
{
    protected $user;
    protected $level;
    protected $category;
    protected $language;
    protected $currency;

    public function __construct(
        CourseLevel $level,
        Category    $category,
        User        $user,
        Currency    $currency,
        Language    $language
    )
    {
        $this->user = $user;
        $this->level = $level;
        $this->currency = $currency;
        $this->language = $language;
        $this->category = $category;
    }

    public function instructorStoreData($playLoad, $user_id)
    {
        if ($user_id) {
            if (gv($playLoad, 'education')) {
                $this->storeEducation(gv($playLoad, 'education'), $user_id);
            }
            if (gv($playLoad, 'certificate')) {
                $this->storeCertificate(gv($playLoad, 'certificate'), $user_id);
            }
            if (gv($playLoad, 'experience')) {
                $this->storeWorkExperience(gv($playLoad, 'experience'), $user_id);
            }
            if (gv($playLoad, 'teachingCategory')) {
                $this->storeTeachingCategory(gv($playLoad, 'teachingCategory'), $user_id);
            }
            if (gv($playLoad, 'teachingLanguage')) {
                $this->storeTeachingLanguage(gv($playLoad, 'teachingLanguage'), $user_id);
            }
            if (gv($playLoad, 'social')) {
                $this->storeSocial(gv($playLoad, 'social'), $user_id);
            }
        }

    }

    private function storeEducation($educations, $user_id)
    {
        $updateEducations = [];
        InstructorEducation::where('instructor_id', $user_id)->delete();

        foreach ($educations as $education) {
            $educationStore = new InstructorEducation();
            $educationStore->instructor_id = $user_id;
            $educationStore->duration = gv($education, 'duration');
            $educationStore->degree = gv($education, 'degree');
            $educationStore->institute = gv($education, 'institute');
            $educationStore->is_verified = gv($education, 'is_verified')
                ? (gv($education, 'is_verified') == 'on' ? 1 : 0) : 0;
            $educationStore->created_by = auth()->user()->id;
            $educationStore->save();
            $updateEducations[] = $educationStore->id;
        }
    }

    private function storeCertificate($certificates, $user_id)
    {
        $updateCertificates = [];
        InstructorCertificate::where('instructor_id', $user_id)->delete();

        foreach ($certificates as $certificate) {
            $certificateStore = new InstructorCertificate();
            $certificateStore->instructor_id = $user_id;
            $certificateStore->duration = gv($certificate, 'duration');
            $certificateStore->title = gv($certificate, 'title');
            $certificateStore->institute = gv($certificate, 'institute');
            $certificateStore->created_by = auth()->user()->id;
            $certificateStore->is_verified = gv($certificate, 'is_verified')
                ? (gv($certificate, 'is_verified') == 'on' ? 1 : 0) : 0;
            $certificateStore->save();
            $updateCertificates[] = $certificateStore->id;
        }
    }

    private function storeWorkExperience($experiences, $user_id)
    {
        $updateExperiences = [];
        InstructorWorkExperience::where('instructor_id', $user_id)->delete();

        foreach ($experiences as $experience) {
            $experienceStore = new InstructorWorkExperience();
            $experienceStore->instructor_id = $user_id;
            $experienceStore->duration = gv($experience, 'duration');
            $experienceStore->designation = gv($experience, 'designation');
            $experienceStore->institute = gv($experience, 'institute');
            $experienceStore->is_verified = gv($experience, 'is_verified')
                ? (gv($experience, 'is_verified') == 'on' ? 1 : 0) : 0;
            $experienceStore->save();
            $updateExperiences[] = $experienceStore->id;
        }
    }

    private function storeTeachingCategory($categories, $user_id)
    {
        InstructorTeachingCategory::where('instructor_id', $user_id)->delete();
        foreach ($categories as $category) {
            $categoryStore = new InstructorTeachingCategory();
            $categoryStore->instructor_id = $user_id;
            $categoryStore->category_id = gv($category, 'category');
            $categoryStore->sub_category_id = gv($category, 'sub_category');
            $categoryStore->level_id = gv($category, 'level');
            $categoryStore->created_by = auth()->user()->id;
            $categoryStore->save();
        }
    }

    private function storeTeachingLanguage($languages, $user_id)
    {
        $updateTeachingLanguage = [];
        InstructorTeachingLanguage::where('instructor_id', $user_id)->delete();

        foreach ($languages as $language) {
            $languageStore = new InstructorTeachingLanguage();
            $languageStore->instructor_id = $user_id;
            $languageStore->language_id = gv($language, 'language');
            $languageStore->level_id = gv($language, 'level');
            $languageStore->created_by = auth()->user()->id;
            $languageStore->save();
            $updateTeachingLanguage[] = $languageStore->id;
        }
    }

    private function storeSocial($socials, $user_id)
    {
        $updateSocial = [];
        InstructorSocial::where('instructor_id', $user_id)->delete();

        foreach ($socials as $social) {
            $socialStore = new InstructorSocial();
            $socialStore->instructor_id = $user_id;
            $socialStore->icon = gv($social, 'icon');
            $socialStore->url = gv($social, 'url');
            $socialStore->save();
            $updateSocial[] = $socialStore->id;
        }
    }

    public function requiredData()
    {
        $data = [];
        $data['levels'] = $this->level->get();
        $data['categories'] = $this->category->get();
        $data['languages'] = $this->language->get();
        return $data;
    }

    public function instructorData()
    {
        $data = $this->requiredData();
        $data['user'] = $this->user->where('id', Auth::id())->with('role')->first();
        $data['currencies'] = $this->currency->whereStatus('1')->get();
        $data['languages'] = $this->language->whereStatus('1')->get();
        $data['countries'] = DB::table('countries')->whereActiveStatus(1)->get();
        $data['states'] = DB::table('states')
            ->where('country_id', $data['user']->country)
            ->where('id', $data['user']->state)
            ->select('id', 'name')
            ->get();
        $data['cities'] = DB::table('spn_cities')
            ->where('state_id', $data['user']->state)
            ->where('id', $data['user']->city)
            ->select('id', 'name')
            ->get();
        return $data;
    }
}
