<?php

namespace Modules\Store\Http\Controllers;

use App\Jobs\SendGeneralEmail;
use App\Jobs\SendNotification;
use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Modules\Setting\Repositories\GeneralSettingRepositoryInterface;
use Modules\Store\Entities\RefundProcess;
use Modules\Store\Entities\RefundRequest;
use Modules\Store\Entities\RefundRequestDetail;
use Modules\Store\Entities\RefundState;
use Yajra\DataTables\Facades\DataTables;
use Modules\Wallet\Http\Controllers\WalletController;

class RefundController extends Controller
{

    protected $generalSettingRepository;

    public function __construct(GeneralSettingRepositoryInterface $generalSettingRepository)
    {
        $this->generalSettingRepository = $generalSettingRepository;
    }


    public function index()
    {
        return view('store::refund.index');
    }

    public function refundsettingUpdate(Request $request)
    {
        $data = [
            'refund_status' => $request->get('refund_status') ? 1 : 0,
            'refund_times' => (int)$request->get('refund_times', 0),
        ];

        $this->generalSettingRepository->update($data);
        Toastr::success(trans('common.Operation successful'), trans('common.Success'));
        return redirect()->back();
    }

    public function all_refund_request_index()
    {
        return view('store::refund.refund_requests.index');
    }

    public function all_refund_request_data(Request $request)
    {
        $data = RefundRequest::with('refund_details', 'refund_details.refund_products', 'order')->latest()->get();

        if ($request->table == 'confirmed') {
            $data = $data->where('is_confirmed', 1);
        } else {
            $data = $data->where('is_confirmed', 0);
        }

        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('date', function ($data) {
                return showDate($data->created_at);
            })
            ->addColumn('email', function ($data) {
                return $data->customer->email;
            })
            ->addColumn('order_id', function ($data) {
                return $data->order->order_number;
            })
            ->addColumn('total_amount', function ($data) {
                return getPriceFormat($data->total_return_amount);
            })
            ->addColumn('request_status', function ($data) {
                if ($data->is_confirmed == 1 && $data->is_confirmed == 1)
                    return '<h6><span class="badge_1">' . __("product.completed") . '</span></h6>';
                elseif ($data->is_confirmed == 1)
                    return '<h6><span class="badge_1">' . __("product.confirmed") . '</span></h6>';
                elseif ($data->is_confirmed == 2)
                    return '<h6><span class="badge_4">' . __("product.declined") . ' </span></h6>';
                else
                    return '<h6><span class="badge_4">' . __("product.pending") . ' </span></h6>';
            })
            ->addColumn('is_refunded', function ($data) {
                if ($data->is_refunded == 1)
                    return '<h6><span class="badge_1">' . __('product.refunded') . '</span></h6>';
                else
                    return '<h6><span class="badge_4">' . __('product.pending') . '</span></h6>';
            })
            ->addColumn('action', function ($data) {
                return view('store::refund.refund_requests.components.refund_action_td', compact('data'));
            })
            ->rawColumns(['request_status', 'is_refunded', 'action'])
            ->toJson();
    }

    public function show($id)
    {
        $data['refund_request'] = RefundRequest::with('refund_details', 'refund_details.refund_products', 'order')->findOrFail($id);
        $data['processes'] = RefundProcess::all();
        return view('store::refund.refund_requests.details', $data);
    }

    public function update_refund_request_by_admin(Request $request, $id)
    {
        try {
            $data = $request->all();
            $refund_request = RefundRequest::findOrFail($id);
            $refund_details = $refund_request->refund_details->first();
            if ($refund_request->is_confirmed != $data['is_confirmed']) {

                if ($data['is_confirmed'] == 1) {
                    $refund_details->update([
                        'processing_state' => 2
                    ]);
                }

            }
            if ($refund_request->is_completed != $data['is_completed']) {

            }
            $refund_request->is_confirmed = $data['is_confirmed'];
            $refund_request->is_completed = $data['is_completed'];

            $refund_request->save();

            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return back();
        } catch (Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return back();
        }
    }

    public function all_refund_request_confirmed_index()
    {
        return view('store::refund.refund_requests.confirmed_index');
    }

    public function seller_refund_request_list()
    {
        $refund_request_details = RefundRequestDetail::with('refund_request', 'seller', 'refund_products', 'order_package')->where('seller_id', auth()->user()->id)->latest()->get();
        return view('store::refund.refund_requests.seller_index', compact('refund_request_details'));
    }

    public function seller_refund_request_data()
    {
        $data = RefundRequestDetail::with('refund_request', 'seller', 'refund_products', 'order_package')->where('seller_id', auth()->id())->latest()->get();
        return DataTables::of($data)
            ->addIndexColumn()
            ->addColumn('date', function ($data) {
                return showDate($data->refund_request->created_at);
            })
            ->addColumn('email', function ($data) {
                return $data->refund_request->customer->email;
            })
            ->addColumn('order_id', function ($data) {
                return $data->refund_request->order->order_number;
            })
            ->addColumn('total_amount', function ($data) {
                return getPriceFormat($data->refund_request->total_return_amount);
            })
            ->addColumn('request_status', function ($data) {
                if ($data->refund_request->is_confirmed == 1 && $data->refund_request->is_confirmed == 1)
                    return '<h6><span class="badge_1">' . __("common.Completed") . '</span></h6>';
                elseif ($data->refund_request->is_confirmed == 1)
                    return '<h6><span class="badge_1">' . __("product.Confirmed") . '</span></h6>';
                elseif ($data->refund_request->is_confirmed == 2)
                    return '<h6><span class="badge_4">' . __("product.declined") . ' </span></h6>';
                else
                    return '<h6><span class="badge_4">' . __("product.pending") . ' </span></h6>';
            })
            ->addColumn('is_refunded', function ($data) {
                if ($data->refund_request->is_confirmed == 2){
                    return '<h6><span class="badge_4">' . __("product.declined") . ' </span></h6>';
                }
                if ($data->refund_request->is_refunded == 1)
                    return '<h6><span class="badge_1">' . __('product.refunded') . '</span></h6>';
                else
                    return '<h6><span class="badge_4">' . __('product.pending') . '</span></h6>';
            })
            ->addColumn('action', function ($data) {
                return view('store::refund.refund_requests.components.my_refund_action_td', compact('data'));
            })
            ->rawColumns(['request_status', 'is_refunded', 'action'])
            ->toJson();
    }

    public function seller_show($id)
    {
        $data['refund_request'] = RefundRequest::with('refund_details', 'refund_details.refund_products', 'order')->findOrFail($id);
        $data['processes'] = RefundProcess::all();
        return view('store::refund.refund_requests.seller_details', $data);
    }

    public function update_refund_state_by_seller(Request $request, $id)
    {
        try {

            $data = $request->all();

            $refund = RefundRequestDetail::findOrFail($id);
            $last_refund_process = RefundProcess::orderByDesc('id')->first();
            $refund->update([
                'processing_state' => $data['processing_state']
            ]);
            if (!$refund->refund_request->is_refunded && $data['processing_state'] == $last_refund_process->id) {

                $refund_infos['seller_id'] = $refund->seller_id;
                $refund_infos['amount'] = $refund->refund_request->total_return_amount;
                $refund_infos['type'] = 'Refund';
                $refund_infos['refund_req_id'] = $refund->refund_request->id;
                $refund_infos['customer_id'] = $refund->refund_request->customer_id;
                $user = User::find($refund_infos['customer_id']);
                if ($refund->refund_request->refund_method == "wallet") {
                    $payment = new WalletController();
                    $response = $payment->returnAmountTowallet($refund_infos);

                }
                SendNotification::dispatch('Store_Refund', $user, [
                    'name' => $user->name,
                    'email' => $user->email,
                    'subject' => 'Store Refund',
                    'message' => 'Refund Money Back to You. Your Money has been returned in your provided bank Account for refund purpose.',
                ],[]);



                $refund->refund_request->update([
                    'is_refunded' => 1,
                    'is_completed' => 1
                ]);
            }

            RefundState::create([
                'refund_request_detail_id' => $id,
                'state' => $data['processing_state']
            ]);


            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return back();
        } catch (Exception $e) {
            Toastr::error(__('common.Something Went Wrong'));
            return back();
        }
    }

}
