<?php

namespace Modules\Survey\Http\Controllers;

use App\Jobs\SurveyNotification;
use App\Traits\SendNotification;
use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Auth;
use Modules\CourseSetting\Entities\Category;
use Modules\CourseSetting\Entities\Course;
use Modules\Org\Entities\OrgBranch;
use Modules\Org\Entities\OrgSurveyBranch;
use Modules\Org\Entities\OrgSurveyPosition;
use Modules\OrgInstructorPolicy\Entities\OrgPolicyCategory;
use Modules\RolePermission\Entities\Role;
use Modules\Survey\Entities\Survey;
use Modules\Survey\Entities\SurveyAssign;
use Modules\Survey\Entities\SurveyGroup;
use Modules\Survey\Entities\SurveyQuestionAssign;
use Modules\Survey\Entities\SurveyQuestionBank;

class SurveyController extends Controller
{
    use SendNotification;

    public function index()
    {
        $categories = Category::where('status', 1)->get();
        $query = Survey::query();

        if (\request()->survey_type) {
            $query->where('survey_type', \request('survey_type'));
        }

        if (\request()->category) {
            $query->where('category_id', \request('category'));
        }
        $surveys = $query->with('participants', 'course', 'user')->latest()->get();
        return view('survey::index', compact('surveys', 'categories'));
    }

    public function create()
    {
        $user = Auth::user();
        if ($user->role_id == 2) {
            $courses = Course::where('type', 1)->where('user_id', Auth::user()->id)->get();
        } else {
            $courses = Course::where('type', 1)->get();
        }


        if (Auth::user()->role_id == 1) {
            $roles = Role::where('id', '!=', 1)->where('id', '!=', 3)->get();
        }
        if (Auth::user()->role_id == 4) {
            $roles = Role::where('id', '!=', 1)->where('id', '!=', 4)->where('id', '!=', 3)->get();
        }
        if (Auth::user()->role_id == 2) {
            $roles = Role::where('id', '!=', 1)->where('id', '!=', 4)->where('id', '!=', 2)->get();
        }

        $query = Category::orderBy('position_order', 'ASC');
        $user = Auth::user();
        if ($user->role_id != 1 && isModuleActive('OrgInstructorPolicy')) {
            $assign = OrgPolicyCategory::where('policy_id', \auth()->user()->policy_id)->pluck('category_id')->toArray();
            $query->whereIn('id', $assign);
        }
        $categories = $query->get();
        $data = [];
        if (isModuleActive('Org')) {
            $data['codes'] = [];
            $data['position_ids'] = [];
        }
        return view('survey::create', $data, compact('courses', 'roles', 'categories'));
    }

    public function studStaffByRole(Request $request)
    {

        try {
            $users = User::where('role_id', $request->id)->get();
            return response()->json([$users]);
        } catch (\Exception $e) {
                        Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function store(Request $request)
    {
        $rules = [
            'title' => 'required',
            'category_id' => 'required',
            'survey_type' => 'required',
        ];

        $request->validate($rules, validationMessage($rules));

        if ($request->survey_type == 1) {
            $rules = [
                'courses' => 'required',
            ];
            $request->validate($rules, validationMessage($rules));
        } else {
            $rules = [
                'deadline_date' => 'required',
                'deadline_time' => 'required',
            ];
            $request->validate($rules, validationMessage($rules));
        }

        if ($request->set_time == 1) {
            $rules = [
                'publish_date' => 'required',
                'publish_time' => 'required',
            ];

            $request->validate($rules, validationMessage($rules));
        }

        try {

            $totalCourse = !empty($request->courses) ? count($request->courses) - 1 : 0;

            for ($i = 0; $i <= $totalCourse; $i++) {
                $survey = new Survey();
                $survey->title = $request->title;
                $survey->survey_type = $request->survey_type;
                $survey->category_id = $request->category_id;
                $survey->survey_for = $request->survey_for;
                $survey->assign_to = $request->assign_to;
                $survey->group_id = $request->group;
                $survey->course_id = $request->courses[$i] ?? 0;
                $survey->description = $request->description;
                $survey->created_by = Auth::id();
                $survey->status = 0;
                if (isModuleActive('Org')) {
                    $survey->org_leaderboard_point = (int)$request->get('org_leaderboard_point', 0);
                }
                if (isset($request->is_anonymous)) {
                    $survey->is_anonymous = 1;
                }
                if (isset($request->set_time)) {
                    $survey->set_time = 1;
                    $survey->publish_date = getPhpDateFormat($request->publish_date);
                    $survey->publish_time = Carbon::parse($request->publish_time)->format("h:i:sa");
                    $survey->is_published = 0;
                } else {
                    $survey->is_published = 1;
                    $survey->publish_date = getPhpDateFormat(date(getActivePhpDateFormat()));
                    $survey->publish_time = date("h:i:sa");
                }
                if ($survey->survey_type != 1) {
                    $survey->deadline_date = getPhpDateFormat($request->deadline_date);
                    $survey->deadline_time = Carbon::parse($request->deadline_time)->format("h:i:sa");
                } else {
                    $survey->deadline_date = null;
                    $survey->deadline_time = null;
                }

                $survey->save();

                if (isModuleActive('Org')) {
                    OrgSurveyBranch::where('survey_id', $survey->id)->delete();
                    OrgSurveyPosition::where('survey_id', $survey->id)->delete();

                    $survey->audience = $request->audience;
                    $survey->position_audience = $request->position_audience;
                    $survey->save();
                    $this->saveOrgSurveyBranch($survey, $request->branch);
                    $this->saveOrgSurveyPosition($survey, $request->position);
                }

            }


            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->route('survey.survey_list');
        } catch (\Exception $e) {
                        Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function assignSurvey($survey, $user)
    {
        $check = SurveyAssign::where('survey_id', $survey->id)->where('user_id', $user->id)->first();
        if (!$check && $survey->survey_type == 1) {
            $survey_assign = new SurveyAssign();
            $survey_assign->survey_id = $survey->id;
            $survey_assign->user_id = $user->id;
            $survey_assign->save();
        }

        $this->sendNotification('Survey_Announcement', $user, [
            'name' => $user->name,
            'surveyTitle' => $survey->title,
            'time' => $survey->deadline,
        ]);


    }

    public function checkAssignSurvey($survey_id, $user_id)
    {
        $survey_assign = SurveyAssign::where('survey_id', $survey_id)->where('user_id', $user_id)->first();
        if (!$survey_assign) {
            $survey_assign = new SurveyAssign();
            $survey_assign->survey_id = $survey_id;
            $survey_assign->user_id = $user_id;
            $survey_assign->save();
        }

    }

    public function deleteAssignSurvey($survey_id, $user_id)
    {
        $survey_assign = SurveyAssign::where('survey_id', $survey_id)->where('user_id', $user_id)->first();
        $survey_assign->delete();

    }


    public function edit($id)
    {
        if (Auth::user()->role_id == 1) {
            $courses = Course::where('type', 1)->get();
        } else {
            $courses = Course::where('type', 1)->where('user_id', Auth::id())->get();
        }


        if (Auth::user()->role_id == 1) {
            $roles = Role::where('id', '!=', 1)->where('id', '!=', 3)->get();
        }
        if (Auth::user()->role_id == 4) {
            $roles = Role::where('id', '!=', 1)->where('id', '!=', 4)->where('id', '!=', 3)->get();
        }
        if (Auth::user()->role_id == 2) {
            $roles = Role::where('id', '!=', 1)->where('id', '!=', 4)->where('id', '!=', 2)->get();
        }
        $edit = Survey::find($id);
        $users = User::wherein('id', explode(',', (int)$edit->users_id))->get();

        $query = Category::orderBy('position_order', 'ASC');
        $user = Auth::user();
        if ($user->role_id != 1 && isModuleActive('OrgInstructorPolicy')) {
            $assign = OrgPolicyCategory::where('policy_id', \auth()->user()->policy_id)->pluck('category_id')->toArray();
            $query->whereIn('id', $assign);
        }
        $categories = $query->get();
        $data = [];

        if (isModuleActive('Org')) {
            $branches = OrgBranch::orderBy('order', 'asc')->with('assignBranchInGroupPolicy');
            $org_policy_branch = OrgSurveyBranch::where('survey_id', $edit->id)->pluck('branch_id')->toArray();
            $data['codes'] = $branches->whereIn('id', $org_policy_branch)->pluck('code')->toArray();
            $data['position_ids'] = OrgSurveyPosition::where('survey_id', $edit->id)->pluck('position_id')->toArray();
        }
        return view('survey::create', $data, compact('courses', 'roles', 'edit', 'users', 'categories'));
    }


    public function update(Request $request)
    {
        $rules = [
            'title' => 'required',
            'category_id' => 'required',
            'survey_type' => 'required',
        ];

        $request->validate($rules, validationMessage($rules));

        if ($request->survey_type == 1) {
            $rules = [
                'courses' => 'required',
            ];
            $request->validate($rules, validationMessage($rules));
        } else {
            $rules = [
                'deadline_date' => 'required',
                'deadline_time' => 'required',
            ];
            $request->validate($rules, validationMessage($rules));
        }

        if ($request->set_time == 1) {
            $rules = [
                'publish_date' => 'required',
                'publish_time' => 'required',
            ];

            $request->validate($rules, validationMessage($rules));
        }

        try {

            $totalCourse = !empty($request->courses) ? count($request->courses) - 1 : 0;

            for ($i = 0; $i <= $totalCourse; $i++) {

                $survey = Survey::find($request->id);
                $survey->title = $request->title;
                $survey->survey_type = $request->survey_type;
                $survey->category_id = $request->category_id;
                $survey->survey_for = $request->survey_for;
                $survey->assign_to = $request->assign_to;
                $survey->group_id = $request->group;
                $survey->course_id = $request->courses[$i] ?? 0;
                $survey->description = $request->description;
                if (isModuleActive('Org')) {
                    $survey->org_leaderboard_point = (int)$request->get('org_leaderboard_point', 0);
                }
                if (isset($request->is_anonymous)) {
                    $survey->is_anonymous = 1;
                } else {
                    $survey->is_anonymous = 0;
                }
                if (isset($request->set_time)) {
                    $survey->set_time = 1;

                    $survey->publish_date = getPhpDateFormat($request->publish_date);
                    $survey->publish_time = Carbon::parse($request->publish_time)->format("h:i:sa");
                    $survey->is_published = 0;
                } else {
                    $survey->is_published = 1;
                    $survey->publish_date = getPhpDateFormat(date(getActivePhpDateFormat()));
                    $survey->publish_time = date("h:i:sa");
                }

                if ($survey->survey_type != 1) {
                    $survey->deadline_date = getPhpDateFormat($request->deadline_date);
                    $survey->deadline_time = Carbon::parse($request->deadline_time)->format("h:i:sa");
                } else {
                    $survey->deadline_date = null;
                    $survey->deadline_time = null;
                }

                $survey->save();

                $this->sendNotificationToUser($survey->id);

                if (isModuleActive('Org')) {
                    OrgSurveyBranch::where('survey_id', $survey->id)->delete();
                    OrgSurveyPosition::where('survey_id', $survey->id)->delete();

                    $survey->audience = $request->audience;
                    $survey->position_audience = $request->position_audience;
                    $survey->save();

                    $this->saveOrgSurveyBranch($survey, $request->branch);
                    $this->saveOrgSurveyPosition($survey, $request->position);
                }
            }
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->route('survey.survey_list');
        } catch
        (\Exception $e) {
                        Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function destroy($id)
    {
        try {
            $survey = Survey::find($id);
            $survey->delete();
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->route('survey.survey_list');
        } catch (\Throwable $th) {
                        Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function setting()
    {
        return view('survey::setting');
    }

    public function settingSubmit(Request $request)
    {
        UpdateGeneralSetting('must_survey_before_certificate', $request->must_survey_before_certificate);
        Toastr::success(trans('common.Operation successful'), trans('common.Success'));
        return redirect()->back();
    }



    public function saveOrgSurveyBranch($survey, $branches): void
    {
        if ($survey->audience != 1) {

            if (!empty($branches)) {
                foreach ($branches as $branch) {
                    $data = [
                        'survey_id' => $survey->id,
                        'branch_id' => $branch
                    ];
                    OrgSurveyBranch::updateOrCreate(
                        $data
                    );
                }
            }
        }
    }

    public function saveOrgSurveyPosition($survey, $positions): void
    {
        if ($survey->position_audience != 1) {
            if (!empty($positions)) {
                foreach ($positions as $key => $position) {
                    if ($position == 1) {
                        $branch = new OrgSurveyPosition();
                        $branch->survey_id = $survey->id;
                        $branch->position_id = $key;
                        $branch->save();
                    }
                }
            }
        }
    }

    public function setQuestion($survey_id)
    {
        try {
            $survey = Survey::find($survey_id);
            $groups = SurveyGroup::all();
            $query = SurveyQuestionBank::query();
            if (!empty(\request('group'))) {
                $query->where('group_id', \request('group'));
            }
            if (!empty(\request('category'))) {
                $query->where('category_id', \request('category'));
            }
            $question_banks = $query->get();
            $already_assigned = SurveyQuestionAssign::where('survey_id', $survey_id)->pluck('question_bank_id')->toArray();
            $categories = Category::where('status', 1)->get();
            return view('survey::question_setup', compact('survey', 'question_banks', 'groups', 'already_assigned', 'categories'));
        } catch (\Throwable $th) {
                        Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function surveyQuestionAssignByAjax(Request $request)
    {
        try {
            SurveyQuestionAssign::where('survey_id', $request->survey_id)->delete();

            if (isset($request->questions)) {
                foreach ($request->questions as $question) {
                    $assign = new SurveyQuestionAssign();
                    $assign->survey_id = $request->survey_id;
                    $assign->question_bank_id = $question;
                    $assign->updated_by = Auth::id();
                    $assign->created_by = Auth::id();
                    $assign->save();
                }

            }

            return response()->json([
                'success' => 'Operation successful',
            ], 200);

        } catch (\Exception $e) {
            return response()->json(['error' => 'Something Went Wrong'], 500);
        }
    }


    public function sendNotificationToUser($survey_id)
    {
        SurveyNotification::dispatch($survey_id);


    }


}
