<?php

namespace Modules\SupportTicket\Http\Controllers;

use App\Traits\SendNotification;
use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Exception;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\CourseSetting\Entities\Course;
use Modules\SupportTicket\Entities\SupportTicket;
use Modules\SupportTicket\Entities\TicketCategory;
use Modules\SupportTicket\Http\Requests\SupportTicketCreateRequest;
use Modules\SupportTicket\Http\Requests\SupportTicketUpdateRequest;
use Modules\SupportTicket\Services\SupportTicketService;
use Yajra\DataTables\Facades\DataTables;

class SupportTicketController extends Controller
{
    use SendNotification;

    protected $supportTicketService;

    public function __construct(SupportTicketService $supportTicketService)
    {
        $this->supportTicketService = $supportTicketService;
    }

    public function index()
    {

        $data['CategoryList'] = $this->supportTicketService->categoryList();
        $data['PriorityList'] = $this->supportTicketService->priorityList();
        $data['StatusList'] = $this->supportTicketService->statusList();
        $data['AgentList'] = $this->supportTicketService->agentList();

        if (isset($_GET['category_id'])) {
            $data['category_id'] = $_GET['category_id'];
        }
        if (isset($_GET['priority_id'])) {
            $data['priority_id'] = $_GET['priority_id'];
        }
        if (isset($_GET['status_id'])) {
            $data['status_id'] = $_GET['status_id'];
        }
        return view('supportticket::ticket.index', $data);
    }


    public function courseTickets()
    {

        try {
            $data['PriorityList'] = $this->supportTicketService->priorityList();
            $data['StatusList'] = $this->supportTicketService->statusList();
            return view('supportticket::ticket.course_tickets', $data);
        } catch (Exception $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function courseTicketsDatatable(Request $request)
    {
        try {
            $TicketList = $this->supportTicketService->coursesTicketList($request->only('f_priority_id', 'f_status_id'));
            $AgentList = User::whereIn('role_id', [1, 2, 4])->get();

            return DataTables::of($TicketList)
                ->addIndexColumn()
                ->addColumn('ticket', function ($TicketList) {
                    return view('supportticket::ticket.components._ticket_td', compact('TicketList'));
                })
                ->addColumn('category', function ($TicketList) {
                    return $TicketList->category->name;
                })
                ->addColumn('username', function ($TicketList) {
                    return view('supportticket::ticket.components._username_td', compact('TicketList'));
                })
                ->addColumn('priority', function ($TicketList) {
                    return $TicketList->priority->name;
                })
                ->addColumn('assign_user', function ($TicketList) {
                    return view('supportticket::ticket.components._assign_user_td', compact('TicketList'));
                })
                ->addColumn('status', function ($TicketList) {
                    return $TicketList->status->name;
                })
                ->addColumn('assign_aggent', function ($TicketList) use ($AgentList) {
                    return view('supportticket::ticket.components._assign_td', compact('TicketList', 'AgentList'));
                })
                ->addColumn('action', function ($TicketList) {
                    return view('supportticket::ticket.components._action_td', compact('TicketList'));
                })
                ->rawColumns(['ticket', 'username', 'assign_user', 'assign_aggent', 'action'])
                ->toJson();

        } catch (Exception $e) {
            return response()->json([
                'error' => $e->getMessage()
            ], 503);
        }


    }

    public function my_ticket(Request $request)
    {

        $data['CategoryList'] = $this->supportTicketService->categoryList();
        $data['PriorityList'] = $this->supportTicketService->priorityList();
        $data['StatusList'] = $this->supportTicketService->statusList();
        $data['AgentList'] = $this->supportTicketService->agentList();

        if ($request->category_id) {
            $data['category_id'] =$request->category_id;
        }
        if ($request->priority_id) {
            $data['priority_id'] = $_GET['priority_id'];
        }
        if ($request->status_id) {
            $data['status_id'] = $_GET['status_id'];
        }

        return view('supportticket::ticket.assigned_ticket_index', $data);
    }

    public function getData()
    {


        $TicketList = $this->supportTicketService->ticketList();
        $AgentList = $this->supportTicketService->agentList();

        return DataTables::of($TicketList)
            ->addIndexColumn()
            ->addColumn('ticket', function ($TicketList) {
                return view('supportticket::ticket.components._ticket_td', compact('TicketList'));
            })
            ->addColumn('category', function ($TicketList) {
                return $TicketList->category->name;
            })
            ->addColumn('username', function ($TicketList) {
                return view('supportticket::ticket.components._username_td', compact('TicketList'));
            })
            ->addColumn('priority', function ($TicketList) {
                return $TicketList->priority->name;
            })
            ->addColumn('assign_user', function ($TicketList) {
                return view('supportticket::ticket.components._assign_user_td', compact('TicketList'));
            })
            ->addColumn('status', function ($TicketList) {
                return $TicketList->status->name;
            })
            ->addColumn('assign_aggent', function ($TicketList) use ($AgentList) {
                return view('supportticket::ticket.components._assign_td', compact('TicketList', 'AgentList'));
            })
            ->addColumn('action', function ($TicketList) {
                return view('supportticket::ticket.components._action_td', compact('TicketList'));
            })
            ->rawColumns(['ticket', 'username', 'assign_user', 'assign_aggent', 'action'])
            ->toJson();
    }

    public function my_ticket_get_data()
    {
        $TicketList = $this->supportTicketService->ticketListMine();

        return DataTables::of($TicketList)
            ->addIndexColumn()
            ->addColumn('ticket', function ($TicketList) {
                return view('supportticket::ticket.components._ticket_td', compact('TicketList'));
            })
            ->addColumn('category', function ($TicketList) {
                return $TicketList->category->name;
            })
            ->addColumn('username', function ($TicketList) {
                return view('supportticket::ticket.components._username_td', compact('TicketList'));
            })
            ->addColumn('priority', function ($TicketList) {
                return $TicketList->priority->name;
            })
            ->addColumn('assign_user', function ($TicketList) {
                return view('supportticket::ticket.components._assign_user_td', compact('TicketList'));
            })
            ->addColumn('status', function ($TicketList) {
                return $TicketList->status->name;
            })
            ->addColumn('action', function ($TicketList) {
                return view('supportticket::ticket.components._action_assigned_td', compact('TicketList'));
            })
            ->rawColumns(['subject'])
            ->rawColumns(['ticket', 'username', 'assign_user', 'action'])
            ->toJson();
    }

    public function store(SupportTicketCreateRequest $request)
    {
        $category = TicketCategory::find($request->category_id);
        $flag = false;
        if ($category->slug == 'course') {
            $request->validate([
                'course_id' => 'required',
            ],
                [
                    'required' => 'This course Field is required',
                ]);
            $flag = true;
        }

        try {
            DB::beginTransaction();
            $this->supportTicketService->create($request->except('_token'));
            DB::commit();

            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            if ($flag) {
                return redirect()->route('ticket.course_tickets.index');
            }
            return redirect()->route('ticket.tickets.index');
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            DB::rollBack();

            return redirect()->back();
        }
    }

    public function create()
    {
        try {

            $data['CategoryList'] = $this->supportTicketService->categoryList();
            $data['PriorityList'] = $this->supportTicketService->priorityList();
            $data['UserList'] = $this->supportTicketService->userList(auth()->id());
            $data['StatusList'] = $this->supportTicketService->statusList();
            $data['AgentList'] = $this->supportTicketService->agentList();

            $user = Auth::user();
            $query = Course::query()->with('user')->where('support', true);
            if ($user->role_id == 5 && isModuleActive('Organization') && Auth::user()->isOrganization()) {
                $query->whereHas('user', function ($q) {
                    $q->where(function ($subQuery) {
                        $subQuery->where('organization_id', Auth::id())
                            ->orWhere('user_id', Auth::id());
                    });
                });
            } elseif ($user->role_id !== 1) {
                $query->where('user_id', $user->id);
            }
            $data['courses'] = $query->get();

            return view('supportticket::ticket.create', $data);
        } catch (Exception $e) {

            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function show($id)
    {
        $data['SupportTicket'] = $this->supportTicketService->supportTicketWithMsgAndFile($id);
        $permission = false;
        $user = auth()->user();
        if ($user->role_id == 1 || $data['SupportTicket']->user_id == $user->id || $data['SupportTicket']->refer_id == $user->id) {
            $permission = true;
        }
        if (!$permission) {
            abort(403);
        }

        $data['SupportTicketStatusList'] = $this->supportTicketService->statusList();
        if ($user->role_id == 1) {
            $data['UserList'] = $this->supportTicketService->userList($data['SupportTicket']->user_id);
            $data['AgentList'] = $this->supportTicketService->agentList();
        }

        return view('supportticket::ticket.show', $data);
    }

    public function edit($id)
    {
        try {
            $data['CategoryList'] = $this->supportTicketService->categoryList();
            $data['PriorityList'] = $this->supportTicketService->priorityList();
            $data['UserList'] = $this->supportTicketService->userList(auth()->id());
            $data['editData'] = $this->supportTicketService->find($id);
            $data['StatusList'] = $this->supportTicketService->statusList();
            $data['AgentList'] = $this->supportTicketService->agentList();
            $user = Auth::user();
            $query = Course::query()->with('user')->where('support', true);
            if ($user->role_id == 5 && isModuleActive('Organization') && Auth::user()->isOrganization()) {
                $query->whereHas('user', function ($q) {
                    $q->where(function ($subQuery) {
                        $subQuery->where('organization_id', Auth::id())
                            ->orWhere('user_id', Auth::id());
                    });
                });
            } elseif ($user->role_id !== 1) {
                $query->where('user_id', $user->id);
            }
            $data['courses'] = $query->get();
            return view('supportticket::ticket.edit', $data);
        } catch (Exception $e) {

            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return redirect()->back();
        }
    }

    public function update(SupportTicketUpdateRequest $request, $id)
    {
        $category = TicketCategory::find($request->category_id);
        $flag = false;
        if ($category->slug == 'course') {
            $request->validate([
                'course_id' => 'required',
            ],
                [
                    'required' => 'This course Field is required',
                ]);
            $flag = true;
        }
        try {
            $this->supportTicketService->update($request->except('_token'), $id);
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            if ($flag) {
                return redirect()->route('ticket.course_tickets.index');
            }
            return redirect()->route('ticket.tickets.index');
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));

            return redirect()->back();
        }
    }

    public function destroy(Request $request)
    {
        try {
            $ticket = $this->supportTicketService->ticketWithFile($request->id);
            $fileIds = [];
            foreach ($ticket->attachFiles as $key => $value) {
                if (file_exists($value->url)) {
                    unlink($value->url);
                }
                $fileIds[] = $value->id;
            }
            $this->supportTicketService->fileDelete($fileIds);
            $ticket->delete();
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->route('ticket.tickets.index');
        } catch (Exception $e) {
            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));

            return redirect()->back();
        }
    }

    public function attachFileDelete(Request $request)
    {
        return $this->supportTicketService->attachFileDelete($request->id);
    }

    public function search()
    {
        $category_id = request()->input('category_id');
        $priority_id = request()->input('priority_id');
        $status_id = request()->input('status_id');



        if (($category_id || $priority_id || $status_id)) {
            $TicketList = $this->supportTicketService->searchWithAttr($category_id, $priority_id, $status_id);
        } else {
            $TicketList = $this->supportTicketService->search();
        }

        $AgentList = $this->supportTicketService->agentList();

        return DataTables::of($TicketList)
            ->addIndexColumn()
            ->addColumn('ticket', function ($TicketList) {
                return view('supportticket::ticket.components._ticket_td', compact('TicketList'));
            })
            ->addColumn('category', function ($TicketList) {
                return $TicketList->category->name;
            })
            ->addColumn('username', function ($TicketList) {
                return view('supportticket::ticket.components._username_td', compact('TicketList'));
            })
            ->addColumn('priority', function ($TicketList) {
                return $TicketList->priority->name;
            })
            ->addColumn('assign_user', function ($TicketList) {
                return view('supportticket::ticket.components._assign_user_td', compact('TicketList'));
            })
            ->addColumn('status', function ($TicketList) {
                return $TicketList->status->name;
            })
            ->addColumn('assign_aggent', function ($TicketList) use ($AgentList) {
                return view('supportticket::ticket.components._assign_td', compact('TicketList', 'AgentList'));
            })
            ->addColumn('action', function ($TicketList) {
                return view('supportticket::ticket.components._action_td', compact('TicketList'));
            })
            ->rawColumns(['ticket', 'username', 'assign_user', 'assign_aggent', 'action'])
            ->toJson();
    }

    public function searchAssigned()
    {
        $category_id = request()->input('category_id');
        $priority_id = request()->input('priority_id');
        $status_id = request()->input('status_id');

        if (!$category_id && !$priority_id && !$status_id) {
            return redirect()->route('ticket.my_ticket');
        }

        $TicketList = $this->supportTicketService->searchAssignedTicketWithAttr($category_id, $priority_id, $status_id);


        return DataTables::of($TicketList)
            ->addIndexColumn()
            ->addColumn('ticket', function ($TicketList) {
                return view('supportticket::ticket.components._ticket_td', compact('TicketList'));
            })
            ->addColumn('category', function ($TicketList) {
                return $TicketList->category->name;
            })
            ->addColumn('username', function ($TicketList) {
                return view('supportticket::ticket.components._username_td', compact('TicketList'));
            })
            ->addColumn('priority', function ($TicketList) {
                return $TicketList->priority->name;
            })
            ->addColumn('assign_user', function ($TicketList) {
                return view('supportticket::ticket.components._assign_user_td', compact('TicketList'));
            })
            ->addColumn('status', function ($TicketList) {
                return $TicketList->status->name;
            })
            ->addColumn('action', function ($TicketList) {
                return view('supportticket::ticket.components._action_assigned_td', compact('TicketList'));
            })
            ->rawColumns(['ticket', 'username', 'assign_user', 'assign_aggent', 'action'])
            ->toJson();
    }

    public function assignedAgent(Request $request)
    {

        $request->validate([
            'ticket_id' => 'required',
            'refer_id' => 'required'
        ]);
        $ticketId = $request->input('ticket_id');
        $agentId = $request->input('refer_id');


        try {

            if ($agentId != 0) {

                // Send Mail
                $refer = User::where('id', $agentId)->first();

                $supportTicket = SupportTicket::find($ticketId);


                $this->sendNotification('ASSIGN_SUPPORT_TICKET', $refer, [
                    'assign_by' => auth()->user()->name,
                    'ticket_no' => $supportTicket->reference_no,
                    'time' => now(),
                    'category' => $supportTicket->category->name,
                    'priority' => $supportTicket->priority->name,
                    'subject' => $supportTicket->subject,
                ]);

                $ticket = $this->supportTicketService->find($ticketId);
                $ticket->refer_id = $agentId;
                $ticket->save();
            } else {
                $ticket = $this->supportTicketService->find($ticketId);
                $ticket->refer_id = 0;
                $ticket->save();
            }
            Toastr::success(trans('common.Operation successful'), trans('common.Success'));
            return redirect()->back();
        } catch (Exception $e) {

            Toastr::error(trans('common.Operation failed'), trans('common.Failed'));
            return abort(404);
        }
    }
}
