<?php

namespace Modules\Appointment\Http\Controllers;

use App\User;
use Brian2694\Toastr\Facades\Toastr;
use Illuminate\Http\Request;
use Illuminate\Routing\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\Appointment\Entities\AppointmentFrontendPage;
use Modules\Appointment\Entities\Booking;
use Modules\Appointment\Entities\Schedule;
use Modules\Appointment\Repositories\Interfaces\ScheduleRepositoryInterface;
use Modules\CourseSetting\Entities\Category;
use Modules\CourseSetting\Entities\CourseLevel;
use Modules\Payment\Entities\Cart;
use Modules\Setting\Model\TimeZone;

class FrontendAppointmentController extends Controller
{
    public $scheduleRepository;

    public function __construct(
        ScheduleRepositoryInterface $scheduleRepository
    )
    {
        $this->scheduleRepository = $scheduleRepository;
    }

    public function index()
    {
        $data['categories'] = Category::withCount('categoryInstructor')->get();
        $data['pages'] = AppointmentFrontendPage::where('status', 1)->get();
        return view(theme('pages.appointment_page'), $data);
    }

    public function instructors(Request $request)
    {
        $data = self::loadData();
        $data['request'] = $request->all();
        return view(theme('pages.appointment_instructor_page'), $data);
    }

    public function instructor($slug)
    {
        $data['slug'] = $slug;
        $data['instructor'] = User::with('teachingLanguages')
            ->withCount('instructorReviews')
            ->where(function ($q) use ($slug) {
                $q->where('slug', $slug)->orWhere('id', $slug);
            })
            ->firstOrFail();

        return view(theme('pages.appointment_single_instructor_page'), $data);
    }

    public function timeZone(Request $request)
    {
        $slug = $request->instructor;
        $next_date = $request->next_date ?? null;
        $pre_date = $request->pre_date ?? null;
        $data = $this->scheduleRepository->datePeriods($next_date, $pre_date);
        $data['timeZones'] = TimeZone::get();
        $data['instructor'] = User::where('slug', $slug)->first(['id', 'slug', 'name']);

        $data['userTimeZone'] = $request->timeZone;
        session()->put('timezone', $request->timeZone);
        return view('appointment::frontend.inc.__view_full_schedule_inner_calendar_body', $data);
    }

    public function checkout(Request $request)
    {

        if (Auth::check()) {
            if (auth()->user()->role_id != 3) {
                Toastr::error(trans('appointment.You are Not Student'), trans('common.Error'));
                return redirect()->back();
            }
        }
        $slot_id = $request->my_slot;
        if ($slot_id) {
            if (Auth::check()) {
                $booked = Booking::where('user_id', auth()->user()->id)
                    ->where('schedule_id', $slot_id)->first();
                if ($booked) {
                    Toastr::warning(trans('appointment.You Already Booked Before!!'), trans('common.Warning'));
                    return redirect()->back();
                }
                $this->storeCart($slot_id);
                $request = '';

                return view(theme('pages.appointment_checkout'), compact('request'));
            } else {
                session(['redirectTo' => route('appointment.checkout', ['my_slot' => $slot_id])]);

                return \redirect(route('login'));
            }
        }
        try {
            if (Auth::check()) {
                $data = [];
                return view('appointment::frontend.checkout', $data);
            } else {
                return view(theme('pages.myCart2'));
            }
        } catch (\Exception $e) {
            GettingError($e->getMessage(), url()->current(), request()->ip(), request()->userAgent());
        }
    }

    public function calendarChange(Request $request)
    {
        $pre_date = $request->pre_date ?? null;
        $next_date = $request->next_date ?? null;
        $data = $this->scheduleRepository->datePeriods($next_date, $pre_date);

        return view('appointment::frontend.inc.__calender_change_button', $data);
    }

    public function categoryInstructor(Request $request, $slug)
    {
        $exploded = explode('-', $slug);
        $category_id = end($exploded);
        $data = self::loadData();
        $data['request'] = $request->merge(['category' => $category_id]);
        return view(theme('pages.appointment_instructor_page'), $data);
    }

    public function storeCart($schedule_id)
    {
        $schedule = Schedule::find($schedule_id);
        Cart::whereNotNull('schedule_id')->delete();
        $cart = new Cart();
        $cart->user_id = auth()->user()->id;
        $cart->instructor_id = $schedule->user_id;
        $cart->schedule_id = $schedule_id;
        $cart->timezone = session()->get('timezone');
        $cart->type = 'appointment';
        $cart->tracking = getTrx();
        if ($schedule->discount_price != null) {
            $cart->price = $schedule->discount_price ?? 0;
        } else {
            $cart->price = $schedule->userInfo->hour_rate ?? 0;
        }
        $cart->save();

    }

    public function reviews($instructor)
    {
        $data['count_5'] = $this->reviewCount($instructor->instructorReviews, 5.00, $instructor->instructor_reviews_count);
        $data['count_4'] = $this->reviewCount($instructor->instructorReviews, 4.00, $instructor->instructor_reviews_count);
        $data['count_3'] = $this->reviewCount($instructor->instructorReviews, 3.00, $instructor->instructor_reviews_count);
        $data['count_2'] = $this->reviewCount($instructor->instructorReviews, 2.00, $instructor->instructor_reviews_count);
        $data['count_1'] = $this->reviewCount($instructor->instructorReviews, 1.00, $instructor->instructor_reviews_count);
    }

    public static function reviewCount($reviews, $echReview, $total)
    {
        $echReviewCount = $reviews->where('star', $echReview)->count();
        if ($echReviewCount > 0) {
            $rating = ($echReviewCount / $total) * 100;
            return $rating;
        }
        return 0;
    }

    public static function loadData()
    {
        $data['categories'] = Category::select('id', 'name', 'title', 'description', 'image', 'thumbnail', 'parent_id')
            ->where('status', 1)
            ->whereNull('parent_id')
            ->withCount('courses')
            ->orderBy('position_order', 'ASC')->with('activeSubcategories', 'childs', 'subcategories')
            ->get();

        $data['languages'] = DB::table('languages')->select('id', 'name', 'code', 'rtl', 'status', 'native')
            ->where('status', 1)
            ->where('lms_id', SaasInstitute()->id)
            ->get();
        $data['levels'] = CourseLevel::get();
        return $data;
    }
}
